
!--------*---------*---------*---------*---------*---------*---------*-*
!     
! Subroutine: eigen_FS
!
! Purpose
! =======
!
! implementation of interface eigen_FS (see eigen_lib.F module)
!
!
! Arguments
! =========
!
! n, nvec, lda, ldz    (input) integer
!  lda and ldz are the leading dimensions corresponding to a and z,
!  respectively. These values must be equal or larger than one obtained
!  by calling eigen_get_matdims.
!
! m_forward, m_backward  (input) integer, optional
!  default values are m_forward = 48, and m_backward = 128.
!
! mode  (input) character, optional
!  = 'A' : eigenvalues and corresponding eigenvectors (default)
!  = 'N' : eigenvalues only
!  = 'X' : mode 'A' + improve accuracy of eigenvalues
!
!--------*---------*---------*---------*---------*---------*---------*-*
      subroutine eigen_FS(n, nvec, a, lda, w, z, ldz,
     &     m_forward, m_backward, mode)

#ifdef WRITE_INPUT_VEC
      use FS_libs_mod
#endif
      use eigen_scaling_mod
      use trbakwy4_mod
      use eigen_trd_mod
      use bisect_mod
      use dc2_FS_mod
      use eigen_libs_mod, only : eigen_get_matdims0,eigen_get_procs
      use eigen_devel_mod
      use comm_mod, only : barrier
      use mpi
!$    use omp_lib
      use FS_libs_mod,only : FS_MYRANK

      implicit none

      integer,   intent(in)           :: n
      integer,   intent(in), optional :: nvec
      real(8),   intent(inout)        :: a(lda,*)
      integer,   intent(in)           :: lda
      real(8),   intent(out)          :: w(*)
      real(8),   intent(out)          :: z(ldz,*)
      integer,   intent(in)           :: ldz
      integer,   intent(in), optional :: m_forward
      integer,   intent(in), optional :: m_backward
      character, intent(in), optional :: mode

      real(8), allocatable   :: d(:), e(:), e2(:)

      integer                :: nvec_, mf, mb
      character              :: mode_

      real(8)                :: SIGMA
      integer                :: nm, ny, nb
      integer                :: my_rank, world_size, ierr
      integer                :: INFO
      logical                :: flag

      real(8)                :: hs0, hs1, s0, s1
      real(8)                :: ret, ret_1, ret_2, ret_3
      integer :: nnod,x_nnod_dummy, y_nnod_dummy
      integer I
#ifdef WRITE_INPUT_VEC
      character*256 :: ofname
#endif

      call eigen_get_procs(nnod,x_nnod_dummy, y_nnod_dummy)
      if( (n<=nnod) .or. (nnod < 4) )then
         call eigen_s0(
     &        n, nvec, a, lda, w, z, ldz,m_forward, m_backward, mode)
         return
      endif

!-
!-------------------------------------------------------------------
!-
      call eigen_get_initialized(flag)
      if (.not. flag) then
        return
      end if
      if (TRD_COMM_WORLD == MPI_COMM_NULL) then
        return
      end if
!-
!-------------------------------------------------------------------
!-
      if (n <= 0) then
        print*,"Warining: Negative dimesion is invalid!"
        return
      end if

      call eigen_get_matdims0(n, nm, ny)
      if (nm <= 0 .or. ny <= 0) then
        print*,"Warining: Problem size is too large for"
     &       // " 32bit fortarn integer biry."
        return
      end if
!-
!-------------------------------------------------------------------
!-
      if (present(nvec)) then
        nvec_ = nvec
      else
        nvec_ = n
      end if
      nvec_ = n

      if (present(mode)) then
        mode_ = mode
      else
        mode_ = 'A'
      end if

      if (present(m_forward)) then
        mf = m_forward
      else
        mf = 48
      end if
      if (mode_ /= 'N') then
        if (present(m_backward)) then
          mb = m_backward
        else
          mb = 128
        end if
      end if
!-
!-------------------------------------------------------------------
!-
      hs0 = eigen_get_wtime()
      ret_1 = ZERO; ret_2 = ZERO; ret_3 = ZERO
!-
!-------------------------------------------------------------------
!-
      if ( mode_ == 'X' ) then
        allocate(d(1:n), e(1:n), e2(1:n))
      else
        allocate(d(1:n), e(1:n))
      endif

      world_size = TRD_nnod
      my_rank    = TRD_inod-1
!-
!-------------------------------------------------------------------
!-
      SIGMA = ONE
!     if (present(mode)) then
!     if (mode == 'X') then
      call eigen_scaling(n, a(1,1), lda, SIGMA)
!     end if
!     end if
!-
!-------------------------------------------------------------------
!-
#if TIMER_PRINT
      s0 = eigen_get_wtime()
#endif

      if (mode_ == 'N' .OR. nvec_ <= 0) then
        call eigen_trd(n, a(1,1), lda, d(1), e(1), mf)
      else
        call eigen_trd(n, a(1,1), lda, w(1), e(1), mf)
      end if
! #ifdef _DEBUGLOG
!       if( FS_MYRANK.eq.0 ) then
!          write(*,*)"+++++++++++++++++++++"
!          DO I=1,N-1
!             write(*,*)"D,E,",I,d(I),e(I+1)
!          ENDDO
!          write(*,*)"D,E,",N,d(N)
!          write(*,*)"+++++++++++++++++++++"
!         write(*,'(a)') "eigen_trd END"
!       endif
! #endif
      ret_1 = dble(n)**3*4/3

#if TIMER_PRINT
      s1 = eigen_get_wtime()
      if (my_rank==0) then
        print 10000,"TRD-BLK ", n, s1-s0, 1D-9*ret_1/(s1-s0), "GFLOPS"
      end if
      call flush(6)
#endif
!-
!-------------------------------------------------------------------
!-
#if TIMER_PRINT
      s0 = eigen_get_wtime()
#endif

      if ((mode_ == 'A' .or. mode_ == 'X') .and. nvec_ > 0) then
#ifdef WRITE_INPUT_VEC
        if (my_rank == 0) then
          write(ofname,'("trdvec",i0,".dat")') n
          open(10,file=ofname,form='unformatted',status='unknown')
          write(10) mat_type
          write(10) n
          write(10) d
          write(10) e
          close(10)
          write(*,*) "output : ", trim(ofname)
          write(*,*) "mattype: ", mat_type
        endif
        goto 99999
#endif
        if (mode_ == 'X') then
          d (1:n) = w(1:n)
          e2(1:n) = e(1:n)
          call dc2_FS(n, nvec_, w(1), e2(2), z(1,1), ldz, INFO, ret_2)
          call eigen_bisect(d(1), e(1), w(1), n, 1)
        else
          d (1:n) = e(1:n)
          call dc2_FS(n, nvec_, w(1), d(2), z(1,1), ldz, INFO, ret_2)
        end if
      end if
      if (mode_ == 'S' ) then
        call eigen_identity(n, z, ldz)
        d(1:n) = w(1:n)
        call eigen_bisect(d(1), e(1), w(1), n, 0)
      else if (mode_ == 'N' .OR. nvec_ <= 0) then
        call eigen_bisect(d(1), e(1), w(1), n, 0)
      end if

#if TIMER_PRINT
      s1 = eigen_get_wtime()
      if (my_rank == 0) then
!       print 10000,"D&C     ", n, s1-s0, 1D-9*ret_2/(s1-s0), "GFLOPS"
        print 10001,"D&C     ", n, s1-s0
      end if
      call flush(6)
#endif
      if (mode_ == 'N' .OR. nvec_ <= 0) then
        goto 99999
      end if
!-
!-------------------------------------------------------------------
!-
#if TIMER_PRINT
      s0 = eigen_get_wtime()
#endif

      nb = 1
      call eigen_common_trbakwy(n, nvec_, a(1,1), lda, z(1,1), ldz,
     &     e, mb, d, nb)
      ret_3 = 2*dble(nvec_)*dble(n)**2

#if TIMER_PRINT
      s1 = eigen_get_wtime()
      if (my_rank == 0) then
        print 10000,"TRDBAK  ", n, s1-s0, 1D-9*ret_3/(s1-s0), "GFLOPS"
      end if
      call flush(6)
#endif
!-
!-------------------------------------------------------------------
!-
      if (SIGMA /= ONE) then
        SIGMA = ONE / SIGMA
        call dscal(n, w, SIGMA, 1)
      end if
!-
!-------------------------------------------------------------------
!-
#if TIMER_PRINT
      hs1 = eigen_get_wtime()
      ret = ret_1 + ret_2 + ret_3
      if (my_rank == 0) then
!       print*,"Total(eigen_FS)",
!    &       hs1-hs0, 1D-9*ret/(hs1-hs0), "GFLOPS"
        print*,"Total(eigen_FS)", hs1-hs0
      end if
      call flush(6)
#endif
!-
!-------------------------------------------------------------------
!-
99999 continue
!-
!-------------------------------------------------------------------
!-
      hs1 = eigen_get_wtime()
      ret = ret_1 + ret_2 + ret_3
      if (ret_2 == 0) ret = -ret
      a(1, 1) = ret
      a(2, 1) = hs1-hs0
#if TIMER_PRINT
      a(3, 1) = comm_time_reduction
     &     + comm_time_dc
     &     + comm_time_backtrafo
#else
      a(3, 1) = -1d0
#endif
!-
!-------------------------------------------------------------------
!-
      if ( mode_ == 'X' ) then
        deallocate(d, e, e2)
      else
        deallocate(d, e)
      end if
!-
!-------------------------------------------------------------------
!-
10000 format (X, A8, I8, E25.16e2, E25.16e2, X, A)
10001 format (X, A8, I8, E25.16e2)
!-
!-------------------------------------------------------------------
!-
      end subroutine eigen_FS
